/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package patch;

import java.io.File;

import com.inet.report.Element;
import com.inet.report.Engine;
import com.inet.report.RDC;
import com.inet.report.ReportException;
import com.inet.report.Section;
import com.inet.report.Subreport;

/**
 * This sample demonstrates how you can modify a large number of reports using API, e.g. change the font of all text
 * fields in the page header section.
 */

//
// Please note: this sample is only a 'how to', not a complete program. To modify
// reports, please add the required java code to the functions patchSection() and patchEngine().
//

public class PatchReports {

    /**
     * Start the patch program.
     * @param args must include the base directory
     */
    public static void main( String[] args ) {
        if( args.length == 0 ) {
            System.out.println( "This is a sample to do changes to reports in a batch job.\n" );
            System.out.println( "Call: java -cp .;core/ClearReports.jar samples.patch.PatchReports <Folder with reports>" );
            return;
        }

        File dir = new File( args[0] );
        if( dir.isDirectory() ) {
            searchDir( dir );
        } else {
            try {
                patchReport( dir );
            } catch( Throwable e ) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Search for rpt files and sub directories. All found rpt files will be patched.
     * @param dir directory
     */
    private static void searchDir( File dir ) {
        File[] files = dir.listFiles();
        if( files == null ) {
            // no directory
            return;
        }
        for( File file : files ) {
            if( file.isDirectory() ) {
                searchDir( file );
            } else {
                if( file.getName().endsWith( ".rpt" ) ) {
                    try {
                        patchReport( file );
                    } catch( Throwable e ) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    /**
     * "Patch" a single rpt file. If there any changes then it will be saved.
     * @param file the report file that should be patched.
     * @throws ReportException if any error occur.
     */
    private static void patchReport( File file ) throws ReportException {
        System.out.println( "Scan File: " + file );
        boolean needSave = false;
        Engine eng = new Engine( Engine.NO_EXPORT );
        eng.setReportFile( file.toString() );

        needSave = patchEngine( eng );

        if( needSave ) {
            RDC.saveEngine( file, eng );
            System.out.println( "\tsave" );
        }
    }

    /**
     * Patch an Engine (main and sub Engine)
     * @param eng the Engine that should be patched.
     * @return true if the Engine was changed.
     * @throws ReportException if any error occurs.
     */
    private static boolean patchEngine( Engine eng ) throws ReportException {
        boolean needSave = false;

        //TODO change some things on a Engine

        for( Section sec : eng.getSections() ) {
            needSave |= patchSection( sec );
        }

        return needSave;
    }

    /**
     * Patch a Section.
     * @param sec the Section that should be patch.
     * @return true if the Engine was changed.
     * @throws ReportException if any error occur.
     */
    private static boolean patchSection( Section sec ) throws ReportException {
        boolean needSave = false;

        //TODO change some things on a Section
        for( Element el : sec.getElements() ) {
            switch( el.getType() ) {
                case Element.FIELD:
                    // FieldElement fieldEl = (FieldElement)el;
                    // TODO change some things on FieldElements
                    break;
                case Element.TEXT:
                    // Text text = (Text)el;
                    // TODO change some things on Text elements
                    break;
                case Element.CROSSTAB:
                    // CrossTab tab = (CrossTab)el;
                    // TODO change some things on crosstabs
                    break;
                case Element.LINE:
                    // Line line = (Line)el;
                    // TODO change some things on lines
                    break;
                case Element.BOX:
                    // Box box = (Box)el;
                    // TODO change some things on boxes
                    break;
                case Element.SUBREPORT:
                    Subreport sub = (Subreport)el;
                    needSave |= patchEngine( sub.getEngine() );
                    break;
            }
        }
        return needSave;
    }
}
